<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Model\Resolver\UrlRewrite;

use Amasty\Blog\Api\CategoryRepositoryInterface;
use Amasty\BlogGraphQl\Model\Resolver\Category as CategoryResolver;
use Amasty\BlogGraphQl\Model\Resolver\UrlRewrite\TypeResolver\Category as CategoryTypeResolver;
use Amasty\BlogGraphQl\Model\UrlRewrite\UrlKeyMatcher;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\Resolver\ContextInterface;
use Magento\Framework\GraphQl\Query\Resolver\Value;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class Category implements ResolverInterface
{
    private const NO_REDIRECT = 0;

    /**
     * @var CategoryRepositoryInterface
     */
    private $categoryRepository;

    /**
     * @var CategoryResolver
     */
    private $categoryResolver;

    /**
     * @var UrlKeyMatcher
     */
    private $urlKeyMatcher;

    public function __construct(
        CategoryRepositoryInterface $categoryRepository,
        CategoryResolver $categoryResolver,
        UrlKeyMatcher $urlKeyMatcher
    ) {
        $this->categoryRepository = $categoryRepository;
        $this->categoryResolver = $categoryResolver;
        $this->urlKeyMatcher = $urlKeyMatcher;
    }

    /**
     * @param Field $field
     * @param ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array|null|Value
     * @throws GraphQlNoSuchEntityException
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $urlKey = $this->urlKeyMatcher->match($args['url'] ?? '');

        if ($urlKey === null) {
            return null;
        }

        $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
        $category = $this->categoryRepository->getByUrlKeyAndStoreId($urlKey, $storeId);

        if (!$category->getCategoryId()) {
            throw new GraphQlNoSuchEntityException(__('Category doesn\'t exist'));
        }

        $category->setData('post_count', $this->getPostCount($category->getCategoryId(), $storeId));

        $result = $category->getData();
        $result[QueryParamInterface::TYPE] = CategoryTypeResolver::AMBLOG_CATEGORY;
        $result[QueryParamInterface::TYPE_ID] = CategoryTypeResolver::AMBLOG_CATEGORY;
        $result[QueryParamInterface::REDIRECT_CODE] = self::NO_REDIRECT;
        $result[QueryParamInterface::RELATIVE_URL] = $args['url'] ?? '';

        return $result;
    }

    private function getPostCount(int $categoryId, int $storeId): int
    {
        $collection = $this->categoryResolver->getCollection();
        $counts = $collection->getPostsCount([$storeId]);

        return (int)($counts[$categoryId] ?? 0);
    }
}
