<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Model\Resolver;

use Amasty\Blog\Api\Data\VoteInterface;
use Amasty\Blog\Model\ResourceModel\Vote\CollectionFactory as VoteCollectionFactory;
use Amasty\Blog\ViewModel\ContentPreparation\Preparers\ProcessWysiwygContent;
use Magento\Framework\App\ObjectManager;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class Post implements ResolverInterface
{
    /**
     * @var ProcessWysiwygContent
     */
    protected $processWysiwygContent;

    /**
     * @var \Amasty\Blog\Api\PostRepositoryInterface
     */
    protected $postRepository;

    /**
     * @var \Amasty\Blog\Api\ViewRepositoryInterface
     */
    protected $viewRepository;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    /**
     * @var \Amasty\Blog\Api\CommentRepositoryInterface
     */
    protected $commentRepository;

    /**
     * @var VoteCollectionFactory
     */
    protected $voteCollectionFactory;

    /**
     * @var \Amasty\Blog\Helper\Settings
     */
    protected $settings;

    public function __construct(
        ProcessWysiwygContent $processWysiwygContent,
        \Amasty\Blog\Api\PostRepositoryInterface $postRepository,
        \Amasty\Blog\Api\ViewRepositoryInterface $viewRepository,
        \Amasty\Blog\Api\CommentRepositoryInterface $commentRepository,
        \Psr\Log\LoggerInterface $logger,
        \Amasty\Blog\Helper\Settings $settings,
        VoteCollectionFactory $voteCollectionFactory = null
    ) {
        $this->processWysiwygContent = $processWysiwygContent;
        $this->postRepository = $postRepository;
        $this->viewRepository = $viewRepository;
        $this->logger = $logger;
        $this->commentRepository = $commentRepository;
        $this->settings = $settings;
        $this->voteCollectionFactory = $voteCollectionFactory
            ?? ObjectManager::getInstance()->get(VoteCollectionFactory::class);
    }

    /**
     * @param Field $field
     * @param \Magento\Framework\GraphQl\Query\Resolver\ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array|\Magento\Framework\GraphQl\Query\Resolver\Value|mixed
     * @throws \Exception
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            if (isset($args['id'])) {
                $post = $this->postRepository->getById($args['id']);
            } else {
                $post = $this->postRepository->getByUrlKey($args['urlKey']);
            }

            $this->viewRepository->create($post->getPostId());
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            throw new GraphQlNoSuchEntityException(__('Wrong parameter.'));
        }

        return $this->convertData($post, $context);
    }

    /**
     * @param string $src
     * @param $context
     *
     * @return string|string[]
     */
    protected function getRelativePath(string $src, $context)
    {
        $baseUrl = trim($context->getExtensionAttributes()->getStore()->getBaseUrl(), '/');

        return str_replace($baseUrl, '', $src);
    }

    /**
     * @param $postId
     *
     * @return mixed
     */
    public function getCommentsCount($postId)
    {
        $commentsCollection = $this->commentRepository->getCommentsInPost($postId)->addActiveFilter();

        return $commentsCollection->getSize();
    }

    public function getVotesCount(int $postId): int
    {
        $voteCollection = $this->voteCollectionFactory->create()->addFieldToFilter(VoteInterface::POST_ID, $postId);

        return $voteCollection->count();
    }

    /**
     * @param $post
     * @param $context
     *
     * @return array
     */
    protected function convertData($post, $context)
    {
        $post->setViews($this->viewRepository->getViewCountByPostId($post->getPostId()));
        $post->setPostThumbnail($this->getRelativePath((string)$post->getPostThumbnailSrc(), $context));
        $post->setListThumbnail($this->getRelativePath((string)$post->getListThumbnailSrc(), $context));
        $post->setData(
            'tag_ids',
            is_array($post->getData('tag_ids')) ? implode(', ', $post->getData('tag_ids')) : $post->getData('tag_ids')
        );
        $post->setData('categories', implode(', ', $post->getData('categories') ?: []));
        $post->setData('comment_count', $this->getCommentsCount($post->getPostId()));
        $post->setData('store_id', implode(', ', $post->getData('stores') ?: []));
        $post->setFullContent($this->processWysiwygContent->prepare($post->getFullContent()));
        $post->setShortContent($this->processWysiwygContent->prepare($post->getShortContent()));
        $post->setData('vote_count', $this->getVotesCount((int)$post->getPostId()));

        return $post->getData();
    }
}
