<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Model\Resolver;

use Amasty\Blog\Api\VoteRepositoryInterface;
use Amasty\Blog\Model\Source\CommentStatus;
use Amasty\Blog\Model\Vote as VoteModel;
use Amasty\Blog\Model\VoteFactory;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;

class LeaveVote implements ResolverInterface
{
    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    /**
     * @var \Magento\Framework\HTTP\PhpEnvironment\RemoteAddress
     */
    private $remoteAddress;

    /**
     * @var VoteFactory
     */
    private $voteFactory;

    /**
     * @var VoteRepositoryInterface
     */
    private $voteRepository;

    public function __construct(
        \Magento\Framework\HTTP\PhpEnvironment\RemoteAddress $remoteAddress,
        VoteFactory $voteFactory,
        VoteRepositoryInterface $voteRepository,
        \Psr\Log\LoggerInterface $logger
    ) {

        $this->logger = $logger;
        $this->remoteAddress = $remoteAddress;
        $this->voteFactory = $voteFactory;
        $this->voteRepository = $voteRepository;
    }

    /**
     * @inheritdoc
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $data = [];

        try {
            $params = $this->extractCommentData($args);
            if ($params['post_id'] > 0 && in_array($params['type'], ['plus', 'minus', 'update'])) {
                $ip = $this->remoteAddress->getRemoteAddress();

                if ($params['type'] != 'update') {
                    $type = ($params['type'] == 'plus') ? '1' : '0';

                    /** @var VoteModel $model */
                    $model = $this->voteRepository->getByIdAndIp($params['post_id'], $ip);
                    $modelType = $model->getType();
                    if ($model->getVoteId()) {
                        $this->voteRepository->delete($model);
                    }

                    if ($modelType === null || $modelType != $type) {
                        $model = $this->voteFactory->create();
                        $model->setIp($ip);
                        $model->setPostId($params['post_id']);
                        $model->setType($type);
                        $this->voteRepository->save($model);
                    }
                }

                $votesForPost = $this->voteRepository->getVotesCount($params['post_id']);
                $voted = $this->voteRepository->getVotesCount($params['post_id'], $ip);
                $data = [
                    'data' => $votesForPost,
                    'voted' => $voted
                ];
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            throw new GraphQlNoSuchEntityException(
                __('Something went wrong. Please review the error log.')
            );
        }

        return $data;
    }

    /**
     * Extract sender data
     *
     * @param array $args
     * @return array
     * @throws GraphQlInputException
     */
    private function extractCommentData(array $args): array
    {
        if (empty($args['input']['type'])) {
            throw new GraphQlInputException(__('Please provide correct Type.'));
        }

        if (empty($args['input']['post_id'])) {
            throw new GraphQlInputException(__('Please provide Post Id.'));
        }

        return [
            'type' => strtolower($args['input']['type']),
            'post_id' => $args['input']['post_id']
        ];
    }
}
