<?php
/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Model\Resolver;

use Amasty\Blog\Model\Source\CommentStatus;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlAuthorizationException;
use Magento\Framework\GraphQl\Exception\GraphQlInputException;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\GraphQl\Model\Query\ContextInterface;

class LeaveComment implements ResolverInterface
{
    /**
     * @var \Amasty\Blog\Helper\Settings
     */
    private $settingsHelper;

    /**
     * @var \Amasty\Blog\Api\PostRepositoryInterface
     */
    private $postRepository;

    /**
     * @var \Amasty\Blog\Api\CommentRepositoryInterface
     */
    private $commentRepository;

    /**
     * @var \Magento\Framework\DataObjectFactory
     */
    private $objectFactory;

    /**
     * @var \Amasty\Blog\Model\Notification\Notification
     */
    private $notificationModel;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    private $logger;

    public function __construct(
        \Amasty\Blog\Helper\Settings $settingsHelper,
        \Amasty\Blog\Api\PostRepositoryInterface $postRepository,
        \Amasty\Blog\Api\CommentRepositoryInterface $commentRepository,
        \Magento\Framework\DataObjectFactory $objectFactory,
        \Amasty\Blog\Model\Notification\Notification $notificationModel,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->settingsHelper = $settingsHelper;
        $this->postRepository = $postRepository;
        $this->commentRepository = $commentRepository;
        $this->objectFactory = $objectFactory;
        $this->notificationModel = $notificationModel;
        $this->logger = $logger;
    }

    /**
     * @inheritdoc
     */
    public function resolve(Field $field, $context, ResolveInfo $info, array $value = null, array $args = null)
    {
        $data = [];

        $commentData = $this->extractCommentData($args);
        $commentData['store_id'] = (int)$context->getExtensionAttributes()->getStore()->getId();
        $postDataObject = $this->objectFactory->create(['data' => $commentData]);

        try {
            $postId = $postDataObject->getPostId();
            $postInstance = $this->postRepository->getById($postId);

            if ($context->getExtensionAttributes()->getIsCustomer()
                || $this->settingsHelper->getCommentsAllowGuests()
            ) {
                $newComment = null;

                $commentModel = $this->commentRepository->getComment();
                $replyTo = $postDataObject->getReplyTo();
                if ($replyTo) {
                    try {
                        $comment = $this->commentRepository->getById($replyTo);
                        $postDataObject->setNewComment('yes');
                        $comment->setReplyTo($comment->getCommentId());
                        $newComment = $this->comment($postDataObject->getData(), $comment, $postInstance);
                    } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                        throw new GraphQlInputException(__('The message for reply wasn`t found'));
                    }
                } else {
                    $postDataObject->unsetData('reply_to');
                    $newComment = $this->comment($postDataObject->getData(), $commentModel, $postInstance);
                }
            } else {
                throw new GraphQlAuthorizationException(__('The current customer isn\'t authorized.'));
            }
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            throw new GraphQlNoSuchEntityException(
                __('Something went wrong. Please review the error log.')
            );
        }

        if ($newComment) {
            $data['comment'] = $newComment->getData();
        }

        return $data;
    }

    /**
     * Extract sender data
     *
     * @param array $args
     * @return array
     * @throws GraphQlInputException
     */
    private function extractCommentData(array $args): array
    {
        if (empty($args['input']['message'])) {
            throw new GraphQlInputException(__('Please provide Message.'));
        }

        if (empty($args['input']['post_id'])) {
            throw new GraphQlInputException(__('Please provide Post Id.'));
        }

        return [
            'name' => $args['input']['name'] ?? '',
            'email' => $args['input']['email'] ?? '',
            'message' => $args['input']['message'],
            'post_id' => $args['input']['post_id'],
            'reply_to' => $args['input']['reply_to'] ?? 0,
        ];
    }

    /**
     * @param array $data
     * @param \Amasty\Blog\Model\Comments $comment
     * @param \Amasty\Blog\Api\Data\PostInterface $postInstance
     * @return \Amasty\Blog\Model\Comments
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    private function comment(array $data, $comment, $postInstance)
    {
        $comment->addData($data);
        if ($comment->getSettingsHelper()->getCommentsAutoapprove()) {
            $comment->setStatus(CommentStatus::STATUS_APPROVED);
            $comment->setSessionId(null);
        } else {
            $comment->setStatus(CommentStatus::STATUS_PENDING);
        }

        if ($comment->getNewComment() == 'yes') {
            $comment->setId(null);
        }

        $comment->setMessage($this->prepareComment(isset($data['message']) ? $data['message'] : ''));
        $this->commentRepository->save($comment);
        try {
            $this->notificationModel->commentNotificationForAdmin((int) $data['store_id'], $comment, $postInstance);
        } catch (\Exception $exception) {
            throw new GraphQlNoSuchEntityException(
                __('Can not send email notification.')
            );
        }

        return $comment;
    }

    /**
     * @param $message
     * @return string
     */
    private function prepareComment($message)
    {
        $message = htmlspecialchars_decode($message);
        $message = strip_tags($message);
        $message = trim($message);

        return $message;
    }
}
