<?php

declare(strict_types=1);

/**
 * @author Amasty Team
 * @copyright Copyright (c) Amasty (https://www.amasty.com)
 * @package Amasty Blog GraphQl for Magento 2 (System)
 */

namespace Amasty\BlogGraphQl\Model\Resolver;

use Amasty\Blog\Model\ResourceModel\Categories\Collection;
use Magento\Framework\GraphQl\Config\Element\Field;
use Magento\Framework\GraphQl\Exception\GraphQlNoSuchEntityException;
use Magento\Framework\GraphQl\Query\ResolverInterface;
use Magento\Framework\GraphQl\Schema\Type\ResolveInfo;
use Magento\Store\Model\Store;

class Category implements ResolverInterface
{
    /**
     * @var null|array
     */
    protected $postsCount = null;

    /**
     * @var Collection
     */
    protected $collection;

    /**
     * @var \Amasty\Blog\Api\CategoryRepositoryInterface
     */
    protected $categoryRepository;

    /**
     * @var \Psr\Log\LoggerInterface
     */
    protected $logger;

    public function __construct(
        \Amasty\Blog\Api\CategoryRepositoryInterface $categoryRepository,
        \Psr\Log\LoggerInterface $logger
    ) {
        $this->categoryRepository = $categoryRepository;
        $this->logger = $logger;
    }

    /**
     * @param Field $field
     * @param \Magento\Framework\GraphQl\Query\Resolver\ContextInterface $context
     * @param ResolveInfo $info
     * @param array|null $value
     * @param array|null $args
     * @return array|\Magento\Framework\GraphQl\Query\Resolver\Value|mixed
     * @throws \Exception
     */
    public function resolve(
        Field $field,
        $context,
        ResolveInfo $info,
        array $value = null,
        array $args = null
    ) {
        try {
            $storeId = (int)$context->getExtensionAttributes()->getStore()->getId();
            $category = $this->categoryRepository->getByIdAndStore($args['id'], $storeId);
        } catch (\Exception $e) {
            $this->logger->error($e->getMessage());
            throw new GraphQlNoSuchEntityException(
                __('Something went wrong. Please review the error log.')
            );
        }

        $category->setData('post_count', $this->getPostsCount($category->getId(), $storeId));
        return $category->getData();
    }

    /**
     * @return Collection
     */
    public function getCollection(): Collection
    {
        if ($this->collection === null) {
            $this->collection = $this->categoryRepository->getActiveCategories();
        }

        return $this->collection;
    }

    /**
     * @param Collection $collection
     */
    public function setCollection(Collection $collection): void
    {
        $this->collection = $collection;
    }

    /**
     * @param int $categoryId
     * @param int $storeId
     *
     * @return int
     */
    protected function getPostsCount($categoryId, int $storeId)
    {
        if ($this->postsCount === null) {
            $this->postsCount = $this->getCollection()->getPostsCount(
                [
                    $storeId,
                    Store::DEFAULT_STORE_ID
                ]
            );
        }

        return isset($this->postsCount[$categoryId]) ? $this->postsCount[$categoryId] : 0;
    }
}
